<?php

namespace Modules\Backup\Tests;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use MicroweberPackages\Core\tests\TestCase;
use Modules\Backup\Backup;
use Modules\Backup\SessionStepper;
use Modules\Content\Models\Content;
use Modules\Post\Models\Post;

/**
 * Run test
 * @author bobi@microweber.com
 * @command php artisan test --filter BackupTest
 */
class GenerateBackupTest extends TestCase
{
    public function testSingleModuleBackup()
    {
        Config::set('microweber.allow_php_files_upload', true);
        $stepsNum = 1115; // very large number on purpose
        $sessionId = SessionStepper::generateSessionId($stepsNum);


        $originalModulePath = modules_path() . 'Logo' . DIRECTORY_SEPARATOR;
        //count the files in path

        $originalModulePathCount = 0;
        //rerucive iterator
        $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($originalModulePath, \RecursiveDirectoryIterator::SKIP_DOTS));
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $originalModulePathCount++;
            }
        }


        for ($i = 0; $i <= $stepsNum; $i++) {
            $backup = new Backup();
            $backup->setSessionId($sessionId);
            $backup->setBackupWithZip(true);
            $backup->setBackupTables(['content']);
            $backup->setBackupModules([
                'Logo',
            ]);
            $status = $backup->start();


            if (isset($status['success'])) {
                break;
            }

        }

        $filepath = $status['data']['filepath'];

        $this->assertNotNull($filepath, 'Filepath not found in status');
        $this->assertTrue(is_file($filepath), 'File not found');

        $zip = new \ZipArchive();
        $zip->open($filepath);


        // list files
        $allFiles = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $allFiles[] = $zip->getNameIndex($i);
        }


        $this->assertNotEmpty($allFiles);

        // Assert files count equals module files + readme + JSON backup file
        $expectedCount = $originalModulePathCount + 2;
        $this->assertEquals($expectedCount, count($allFiles), 'File count mismatch - expected ' . $expectedCount . ' files');

        $moduleInZip1 = $zip->getFromName('Modules/Logo/module.json');
        $moduleInZip2 = $zip->getFromName('Modules/Logo/Filament/LogoModuleSettings.php');
        $this->assertNotEmpty($moduleInZip1);
        $this->assertNotEmpty($moduleInZip2);
        $zip->close();


    }

    public function testSingleTableBackup()
    {

        $getAllContent = Content::all();
        $getAllContent->each(function ($content) {
            $content->delete();
        });

        $post = new Post();
        $post->url = 'test-post';
        $post->title = 'Test post';
        $post->save();

        $sessionId = SessionStepper::generateSessionId(1);

        $backup = new Backup();
        $backup->setSessionId($sessionId);
        $backup->setAllowSkipTables(false);
        $backup->setBackupTables(['content']);
        $backup->setBackupMedia(false);
        $backup->setBackupModules([]);
        $backup->setBackupTemplates([]);

        $status = $backup->start();

        // Get filepath from the data array
        $filepath = $status['data']['filepath'];

        $this->assertNotNull($filepath, 'Filepath not found in status data');
        $content = file_get_contents($filepath);
        $content = json_decode($content, true);

        $this->assertNotEmpty($content['content'][0]['url']);
        $this->assertNotEmpty($content['__table_structures']['content']['url']);

        $this->assertSame($content['content'][0]['url'], 'test-post');
        $this->assertSame($content['content'][0]['title'], 'Test post');


    }





    public function testUserfilesOneStepTest()
    {
        Config::set('microweber.allow_php_files_upload', true);
        // Use a single step for the backup
        $stepsNum = 1;
        $sessionId = SessionStepper::generateSessionId($stepsNum);

      //  $originalFilesPath = userfiles_path();

        $originalFilesPath = storage_path('app/public/');

        //count the files in path

        $originalFilesPathCount = 0;
        //recursive iterator with same settings as in ZipBatchBackup class
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($originalFilesPath, \RecursiveDirectoryIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if (!$file->isDir()) {
                $originalFilesPathCount++;
            }
        }

        // Run the backup
        $backup = new Backup();
        $backup->setSessionId($sessionId);
        $backup->setBackupWithZip(true);
        $backup->setBackupAllData(true);
        $backup->setBackupTables(['content']);

        // First step
        $status = $backup->start();


        // If we need to run a second step
        if (isset($status['data']) && $status['data'] === false) {
            // Run the second step to complete the backup
            $status = $backup->start();
        }

        $this->assertTrue(isset($status['data']), 'Status data not found');
        $this->assertTrue(isset($status['data']['filepath']), 'Filepath not found in status data');

        $filepath = $status['data']['filepath'];
        $this->assertNotNull($filepath, 'Filepath is null');
        $this->assertTrue(is_file($filepath), 'File not found at: ' . $filepath);

        $zip = new \ZipArchive();
        $zip->open($filepath);

        // List files in the zip
        $allFiles = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $allFiles[] = $zip->getNameIndex($i);
        }
        $this->assertNotEmpty($allFiles);

        // Check that we have files in the backup
        $this->assertGreaterThan(0, count($allFiles), 'Backup should contain files');
        $this->assertEquals($originalFilesPathCount + 1, count($allFiles));

    }

    public function testMediaBackupThreeStepTest()
    {
        Config::set('microweber.allow_php_files_upload', true);
        // Use three steps for the backup
        $stepsNum = 3;
        $sessionId = SessionStepper::generateSessionId($stepsNum);

      //  $originalFilesPath = userfiles_path();
        //count the files in path
        $originalFilesPath = storage_path('app/public/');

        $originalFilesPathCount = 0;
        //recursive iterator with same settings as in ZipBatchBackup class
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($originalFilesPath, \RecursiveDirectoryIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if (!$file->isDir()) {
                $originalFilesPathCount++;
            }
        }

        // We need to run the backup for all steps - this is a multi-step backup
        // Unlike a single-step backup, we need to iterate until completion
        $status = null;
        for ($i = 0; $i < $stepsNum; $i++) {
            $backup = new Backup();
            $backup->setSessionId($sessionId);
            $backup->setBackupWithZip(true);
            $backup->setBackupAllData(true);
            $backup->setBackupTables(['content']);

            $status = $backup->start();

            // If success is set, we're done
            if (isset($status['success'])) {
                break;
            }
        }

        $this->assertTrue(isset($status['data']), 'Status data not found');
        $this->assertTrue(isset($status['data']['filepath']), 'Filepath not found in status data');

        $filepath = $status['data']['filepath'];
        $this->assertNotNull($filepath, 'Filepath is null');
        $this->assertTrue(is_file($filepath), 'File not found at: ' . $filepath);

        $zip = new \ZipArchive();
        $zip->open($filepath);

        // List files in the zip
        $allFiles = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $allFiles[] = $zip->getNameIndex($i);
        }
        $this->assertNotEmpty($allFiles);

        // Check that we have files in the backup
        $this->assertGreaterThan(0, count($allFiles), 'Backup should contain files');
        $this->assertEquals($originalFilesPathCount + 1, count($allFiles));


    }

    public function testBackupOnAllTablesTest()
    {
        Config::set('microweber.allow_php_files_upload', true);
        $stepsNum = 5; // Use multiple steps to ensure complete backup
        $sessionId = SessionStepper::generateSessionId($stepsNum);

        // Get all database tables for verification later
        $allDbTables = app()->database_manager->get_tables_list();
        $this->assertNotEmpty($allDbTables, 'No database tables found');

        // Count records in each table before backup
        $originalTableCounts = [];
        foreach ($allDbTables as $table) {
            $originalTableCounts[$table] = DB::table($table)->count();
        }

        $status = null;
        for ($i = 0; $i < $stepsNum; $i++) {
            $backup = new Backup();
            $backup->setSessionId($sessionId);
            $backup->setBackupWithZip(true);
            $backup->setBackupAllData(true);
            $backup->setAllowSkipTables(false); // Don't skip any tables

            $status = $backup->start();

            // If success is set, we're done
            if (isset($status['success'])) {
                break;
            }
        }

        $this->assertTrue(isset($status['success']), 'Backup process did not complete successfully');
        $this->assertTrue(isset($status['data']['filepath']), 'Filepath not found in status data');

        $filepath = $status['data']['filepath'];
        $this->assertNotNull($filepath, 'Filepath is null');
        $this->assertTrue(is_file($filepath), 'Backup file not found at: ' . $filepath);

        // Open the zip file to verify its contents
        $zip = new \ZipArchive();
        $zip->open($filepath);

        // Check for backup.json which contains database data
        $backupContent = $zip->getFromName('backup.json');
        $this->assertNotEmpty($backupContent, 'backup.json not found in zip archive');

        // Parse backup content
        $backupData = json_decode($backupContent, true);
        $this->assertNotEmpty($backupData, 'Backup data is empty');

        // Verify table structures exist
        $this->assertArrayHasKey('__table_structures', $backupData, 'Table structures not found in backup');

        // Check that at least some essential tables are included
        $essentialTables = ['content', 'categories', 'options'];
        foreach ($essentialTables as $table) {
            $this->assertArrayHasKey($table, $backupData['__table_structures'], "Table '$table' structure not found in backup");
        }

        // Verify the table count matches the original database
        $backupTableCount = count($backupData['__table_structures']);
        $originalTableCount = count($allDbTables);

        $this->assertGreaterThan(0, $backupTableCount, 'No tables found in the backup');
        $this->assertEquals($originalTableCount, $backupTableCount,
            'Backup contains different table count than the original database');

        // Verify record counts for each table
        $recordCountMismatches = [];
        foreach ($allDbTables as $table) {
            // Skip the __table_structures entry as it's not a real table
            if ($table === '__table_structures') {
                continue;
            }

            // Check if the table exists in backup data
            if (isset($backupData[$table])) {
                $backupRecordCount = count($backupData[$table]);
                $originalRecordCount = $originalTableCounts[$table];

                // If counts don't match, add to mismatches list
                if ($backupRecordCount != $originalRecordCount) {
                    $recordCountMismatches[$table] = [
                        'original' => $originalRecordCount,
                        'backup' => $backupRecordCount
                    ];
                }
            }
        }

        $this->assertEmpty($recordCountMismatches,
            'Record count mismatches found: ' . json_encode($recordCountMismatches));

        $zip->close();
    }

}
