<?php

namespace Modules\Updater\Services;

class UpdaterHelper
{
    /**
     * Get the latest version from the update server
     */
    public function getLatestVersion($selectedBranch = 'master')
    {
        $key = 'standalone_updater_latest_version__'.$selectedBranch;
      return cache()->remember($key, 1440, function () use ($selectedBranch) {
            $updateApi = 'http://updater.microweberapi.com/builds/' . $selectedBranch . '/version.txt';

            $version = app()->url_manager->download($updateApi);
            if ($version) {
                $version = trim($version);
                return $version;
            }
            return MW_VERSION;
       });
    }

    /**
     * Get messages about why the system can't be updated
     */
    public function getCanUpdateMessages(): array
    {

        $messages = [];
        $projectMainDir = base_path();

        if (is_dir($projectMainDir . DS . '.git')) {
            $messages[] = 'A .git folder is present on your server. Please remove it before updating.';
        }

        if (!class_exists('ZipArchive')) {
            $messages[] = 'ZipArchive PHP extension is required auto updater.';
        }

        if (!function_exists('curl_init')) {
            $messages[] = 'The Curl PHP extension is required auto updater.';
        }

        if (!function_exists('json_decode')) {
            $messages[] = 'The JSON PHP extension is required auto updater.';
        }

        if (!is_writable($projectMainDir . DS . 'src')) {
            $messages[] = 'The src folder must be writable.';
        }

        if (!is_writable(userfiles_path())) {
            $messages[] = 'The userfiles folder must be writable.';
        }

        if (!is_writable(storage_path())) {
            $messages[] = 'The storage folder must be writable.';
        }

        if (!is_writable(public_path())) {
            $messages[] = 'The public folder must be writable.';
        }

        if (is_link($projectMainDir . DS . 'vendor')) {
            $messages[] = 'The vendor folder must not be a symlink.';
        }

        if (!is_writable($projectMainDir . DS . 'vendor')) {
            $messages[] = 'The vendor folder must be writable.';
        }

        if (function_exists('disk_free_space')) {
            $bytes = disk_free_space($projectMainDir);
            $si_prefix = array('B', 'KB', 'MB', 'GB', 'TB', 'EB', 'ZB', 'YB');
            $base = 1024;
            $class = min((int)log($bytes, $base), count($si_prefix) - 1);

            if (($bytes / pow($base, $class)) < 1) {
                $messages[] = 'The minimum required free disk space is 1GB, you have ' . sprintf('%1.2f', $bytes / pow($base, $class)) . ' ' . $si_prefix[$class] . ' on your server.';
            }
        }

        return $messages;
    }

    /**
     * Copy the standalone updater files to the public directory
     */
    public function copyStandaloneUpdater($updateCacheDir,$skipUi=false)
    {

        if(!is_dir($updateCacheDir)){
            mkdir_recursive($updateCacheDir);
        }


        // Clear bootstrap cache
        $bootstrap_cached_folder = normalize_path(base_path('bootstrap/cache/'), true);
        rmdir_recursive($bootstrap_cached_folder);

        // Copy files from the new stubs location
        $stubsPath = module_path('Updater') . DS . 'Stubs';

        // Create the standalone-updater.php file which contains all necessary code
        $standaloneUpdaterContent = $this->generateStandaloneUpdaterFile($stubsPath,$skipUi);
        file_put_contents($updateCacheDir . DS . 'index.php', $standaloneUpdaterContent);

        return true;
    }

    /**
     * Generate the standalone updater file content
     */
    public function generateStandaloneUpdaterFile($stubsPath, $skipUi = false)
    {
        // Read the source files
        $actionsContent = file_get_contents($stubsPath . DS . 'actions.source.php.stub');
        $indexContent = file_get_contents($stubsPath . DS . 'index.source.php.stub');
        $unzipContent = file_get_contents($stubsPath . DS . 'Unzip.source.php.stub');
        $executorContent = file_get_contents($stubsPath . DS . 'StandaloneUpdateExecutor.source.php.stub');
        $replacerContent = file_get_contents($stubsPath . DS . 'StandaloneUpdateReplacer.source.php.stub');


        // Create the standalone updater file
        $standaloneUpdaterContent = <<<EOT
<?php
/**
 * Standalone Updater for Microweber
 *
 * This file is used to update Microweber without the need for the admin panel.
 * Just place this file in the root directory of your Microweber installation and run it.
 */

// Set error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
define("INI_SYSTEM_CHECK_DISABLED", ini_get('disable_functions'));

if (!strstr(INI_SYSTEM_CHECK_DISABLED, 'ini_set')) {
    ini_set("memory_limit", "512M");
    ini_set("set_time_limit", 0);
}

if (!strstr(INI_SYSTEM_CHECK_DISABLED, 'date_default_timezone_set')) {
    date_default_timezone_set('UTC');
}

// Define the Unzip class
{$unzipContent}

// Define the StandaloneUpdateExecutor class
{$executorContent}

// Define the StandaloneUpdateReplacer class
{$replacerContent}

// Handle actions
{$actionsContent}

// Display the UI

EOT;


        if(!$skipUi){
            $standaloneUpdaterContent .= <<<EOT

?>
{$indexContent}

EOT;

        }

        return $standaloneUpdaterContent;
    }


}
