<?php

namespace Modules\Content\Repositories;

use Content;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Collection;
use Modules\Content\Support\BreadcrumbLinks;
use Modules\Content\Support\ContentManagerCrud;
use Modules\Content\Support\ContentManagerHelpers;
use Modules\Content\Support\PagesTree;
use Modules\Content\Support\PagingLinks;
use Modules\Content\Support\PagingNav;


/**
 * Content class is used to get and save content in the database.
 *
 * @category Content
 * @desc     These functions will allow you to get and save content in the database.
 */
class ContentManager
{
    public static $skip_pages_starting_with_url = ['admin', 'api', 'module'];
    public $tables = array();
    public $table_prefix = false;

    /** @var \MicroweberPackages\App\LaravelApplication */
    public $app;

    /** @var ContentManagerCrud */
    public $crud;

    /** @var ContentManagerHelpers */
    public $helpers;


    public $content_id = false;
    public $product_id = false;
    public $page_id = false;
    public $main_page_id = false;
    public $post_id = false;
    public $category_id = false;


    /**
     *  Boolean that indicates the usage of cache while making queries.
     *
     * @var
     */
    public $no_cache = false;
    public $contentRepository;
    public $pagingLinks;
    public $pagesTree;
    public $breadcrumbLinks;
    public $pagingNav;


    public function __construct($app = null)
    {
        if (!is_object($this->app)) {
            if (is_object($app)) {
                $this->app = $app;
            } else {
                $this->app = mw();
            }
        }


        $this->crud = new ContentManagerCrud($this->app);
        $this->helpers = new ContentManagerHelpers($this->app);
        $this->contentRepository = new ContentRepository();
        $this->pagingLinks = new PagingLinks($this->app);
        $this->pagesTree = new PagesTree($this->app);
        $this->breadcrumbLinks = new BreadcrumbLinks($this->app);
        $this->pagingNav = new PagingNav($this->app);
        //$this->content_repository = $this->app->repository_manager->driver(\MicroweberPackages\Content\Content::class);


    }


    function post_id()
    {
        return $this->app->template_manager->getPostId();
    }


    function product_id()
    {
        return $this->app->template_manager->getProductId();
    }

    function content_id()
    {
        return $this->app->template_manager->getContentId();

    }

    function category_id()
    {
        return $this->app->template_manager->getCategoryId();

    }

    function page_id()
    {
        return $this->app->template_manager->getPageId();
    }

    function main_page_id()
    {
        return $this->app->template_manager->getMainPageId();
    }


    /**
     * Get single content item by id from the content_table.
     *
     * @param int|string $id The id of the page or the url of a page
     *
     * @return array The page row from the database
     *
     * @category  Content
     * @function  get_page
     *
     * @example
     * <pre>
     * Get by id
     * $page = $this->get_page(1);
     * var_dump($page);
     * </pre>
     */
    public function get_page($id = 0)
    {
        if (intval($id) == 0) {
            return false;
        }

        $page = $this->get_by_id($id);
        return $page;


    }

    /**
     * Get single content item by id from the content_table.
     *
     * @param int $id The id of the content item
     *
     * @return array
     *
     * @category  Content
     * @function  get_content_by_id
     *
     * @example
     * <pre>
     * $content = $this->get_by_id(1);
     * var_dump($content);
     * </pre>
     */
    public function get_by_id($id)
    {

        return app()->content_repository->getById($id);

    }

    public function get_by_url($url = '', $no_recursive = false)
    {
        return $this->crud->get_by_url($url, $no_recursive);
    }

    public function get_by_title($title = '')
    {
        return $this->get(['single' => true, 'title' => $title]);
    }

    public function get_content_id_from_url($url = '')
    {
        $content = $this->get_by_url($url);
        if ($content && isset($content['id'])) {
            return $content['id'];
        }
    }


    /**
     * Get array of content items from the database.
     *
     * It accepts string or array as parameters. You can pass any db field name as parameter to filter content by it.
     * All parameter are passed to the get() function
     *
     * You can get and filter content and also order the results by criteria
     *
     * @function get_content
     *
     *
     * @desc     Get array of content items from the content DB table
     *
     * @param mixed|array|bool|string $params You can pass parameters as string or as array
     * @params
     *
     * *Some parameters you can use*
     *  You can use all defined database fields as parameters
     *
     * .[params-table]
     *|-----------------------------------------------------------------------------
     *| Field Name          | Description               | Values
     *|------------------------------------------------------------------------------
     *| id                  | the id of the content     |
     *| is_active           | published or unpublished  | "y" or "n"
     *| parent              | get content with parent   | any id or 0
     *| created_by          | get by author id          | any user id
     *| created_at          | the date of creation      |
     *| updated_at          | the date of last edit     |
     *| content_type        | the type of the content   | "page" or "post", anything custom
     *| subtype             | subtype of the content    | "static","dynamic","post","product", anything custom
     *| url                 | the link to the content   |
     *| title               | Title of the content      |
     *| content             | The html content saved in the database |
     *| description         | Description used for the content list |
     *| position            | The order position        |
     *| active_site_template   | Current template for the content |
     *| layout_file         | Current layout from the template directory |
     *| is_deleted          | flag for deleted content  |  "n" or "y"
     *| is_home             | flag for homepage         |  "n" or "y"
     *| is_shop             | flag for shop page        |  "n" or "y"
     *
     * @return array|bool|mixed Array of content or false if nothing is found
     *
     * @uses     get() You can use all the options of get(), such as limit, order_by, count, etc...
     *
     * @example
     * #### Get with parameters as array
     * <code>
     *
     * $params = array();
     * $params['is_active'] = 1; //get only active content
     * $params['parent'] = 2; //get by parent id
     * $params['created_by'] = 1; //get by author id
     * $params['content_type'] = 'post'; //get by content type
     * $params['subtype'] = 'product'; //get by subtype
     * $params['title'] = 'my title'; //get by title
     *
     * $data = $this->get($params);
     * var_dump($data);
     *
     * </code>
     * @example
     * #### Get by params as string
     * <code>
     *  $data = $this->get('is_active=1');
     *  var_dump($data);
     * </code>
     * @example
     * #### Ordering and sorting
     * <code>
     *  //Order by position
     *  $data = $this->get('content_type=post&is_active=1&order_by=position desc');
     *  var_dump($data);
     *
     *  //Order by date
     *  $data = $this->get('content_type=post&is_active=1&order_by=updated_at desc');
     *  var_dump($data);
     *
     *  //Order by title
     *  $data = $this->get('content_type=post&is_active=1&order_by=title asc');
     *  var_dump($data);
     *
     *  //Get content from last week
     *  $data = $this->get('created_at=[mt]-1 week&is_active=1&order_by=title asc');
     *  var_dump($data);
     * </code>
     */
    public function get($params = false)
    {
        return $this->crud->get($params);
    }

    public function get_children($id = 0)
    {
        return app()->content_repository->getChildren($id);
    }

    public function get_data($params = false)
    {
        return $this->app->data_fields_manager->get($params);
    }

    public function data($content_id, $field_name = false)
    {
        $data = array();
        $data['content_id'] = intval($content_id);
        $values = $this->app->data_fields_manager->getValues($data);

        if ($field_name) {
            if (isset($values[$field_name])) {
                return $values[$field_name];
            } else {
                return false;
            }
        }

        return $values;
    }

    public function tags($content_id = false, $return_full = false)
    {
        return $this->app->content_repository->tags($content_id, $return_full);
    }

    public function attributes($content_id)
    {
        $data = array();
        $data['rel_type'] = morph_name(\Modules\Content\Models\Content::class);
        $data['rel_id'] = intval($content_id);

        return $this->app->attributes_manager->getValues($data);
    }

    /**
     * paging.
     *
     * paging
     *
     * @param $params ['num'] = 5; //the numer of pages
     *
     * @return string - html string with ul/li
     * @link
     *
     * @category  posts
     *
     * @internal  param $display =
     *            'default' //sets the default paging display with <ul> and </li>
     *            tags. If $display = false, the function will return the paging
     *            array which is the same as $posts_pages_links in every template
     *
     * @author    Microweber
     *
     */
    public function paging($params)
    {


        return $this->pagingNav->get($params);

    }

    public function paging_links($base_url = false, $pages_count = false, $paging_param = 'current_page', $keyword_param = 'keyword')
    {

        return $this->pagingLinks->get($base_url, $pages_count, $paging_param, $keyword_param);
    }

    /**
     * Print nested tree of pages.
     *
     * @param int $parent
     * @param bool $link
     * @param bool $active_ids
     * @param bool $active_code
     * @param bool $remove_ids
     * @param bool $removed_ids_code
     * @param bool $ul_class_name
     * @param bool $include_first
     *
     * @return sting Prints the pages tree
     * @example
     * <pre>
     * // Example Usage:
     * $pt_opts = array();
     * $pt_opts['link'] = "{title}";
     * $pt_opts['list_tag'] = "ol";
     * $pt_opts['list_item_tag'] = "li";
     * pages_tree($pt_opts);
     * </pre>
     * @example
     * <pre>
     * // Example Usage to make <select> with <option>:
     * $pt_opts = array();
     * $pt_opts['link'] = "{title}";
     * $pt_opts['list_tag'] = " ";
     * $pt_opts['list_item_tag'] = "option";
     * $pt_opts['active_ids'] = $data['parent'];
     * $pt_opts['active_code_tag'] = '   selected="selected"  ';
     * $pt_opts['ul_class'] = 'nav';
     * $pt_opts['li_class'] = 'nav-item';
     *  pages_tree($pt_opts);
     * </pre>
     * @example
     * <pre>
     * // Other opltions
     * $pt_opts['parent'] = "8";
     * $pt_opts['include_first'] =  true; //includes the parent in the tree
     * $pt_opts['id_prefix'] = 'my_id';
     * </pre>
     *
     */
    public function pages_tree($parent = 0, $link = false, $active_ids = false, $active_code = false, $remove_ids = false, $removed_ids_code = false, $ul_class_name = false, $include_first = false)
    {
        $params2 = array();
        $params = false;
        $output = '';
        if (is_integer($parent)) {
        } else {
            $params = $parent;
            if (is_string($params)) {
                $params = parse_str($params, $params2);
                $params = $params2;
                extract($params);
            }
            if (is_array($params)) {
                $parent = 0;
                extract($params);
            }
        }
        if (!defined('CONTENT_ID')) {
            $this->define_constants();
        }

        $cache_id_params = $params;
        if (isset($cache_id_params['link']) and is_callable($cache_id_params['link'])) {
            unset($cache_id_params['link']);
            $params['no_cache'] = true;
        }


        $function_cache_id = false;
        $args = func_get_args();
        foreach ($args as $k => $v) {
            $function_cache_id = $function_cache_id . serialize($k) . serialize($v);

        }
        $function_cache_id = $function_cache_id . serialize($cache_id_params);

        $function_cache_id = __FUNCTION__ . crc32($function_cache_id) . PAGE_ID . $parent;
        if ($parent == 0) {
            $cache_group = 'content/global';
        } else {
            $cache_group = 'categories/global';
        }
        if (isset($include_categories) and $include_categories == true) {
            $cache_group = 'categories/global';
        }

        $nest_level = 0;

        if (isset($params['nest_level'])) {
            $nest_level = $params['nest_level'];
        }

        $nest_level_orig = $nest_level;
        //$params['no_cache'] = 1;
        if ($nest_level_orig == 0) {
//            $cache_content = $this->app->cache_manager->get($function_cache_id, $cache_group);
//            if (isset($params['no_cache'])) {
//                $cache_content = false;
//            }
//            // @todo: activate cache
//            $cache_content = false;
//            if (($cache_content) != false) {
//                if (isset($params['return_data'])) {
//                    return $cache_content;
//                } else {
//                    echo $cache_content;
//                }
//
//                return;
//            }
        }

        $nest_level = 0;

        if (isset($params['nest_level'])) {
            $nest_level = $params['nest_level'];
        }
        if (isset($params['parent'])) {
            $params['parent'] = intval($params['parent']);
        }

        if (isset($params['id'])) {
            unset($params['id']);
        }

        $max_level = false;
        if (isset($params['max_level'])) {
            $max_level = $params['max_level'];
        } elseif (isset($params['maxdepth'])) {
            $max_level = $params['max_level'] = $params['maxdepth'];
        } elseif (isset($params['depth'])) {
            $max_level = $params['max_level'] = $params['depth'];
        }

        if ($max_level != false) {
            if (intval($nest_level) >= intval($max_level)) {
                echo '';

                return;
            }
        }

        $is_shop = '';
        if (isset($params['is_shop'])) {
            if ($params['is_shop'] == 'y') {
                $params['is_shop'] = 1;
            } elseif ($params['is_shop'] == 'n') {
                $params['is_shop'] = 0;
            }

            $is_shop = $this->app->database_manager->escape_string($params['is_shop']);
            $is_shop = " and is_shop='{$is_shop} '";
            $include_first = false;
        }
        $ul_class = 'pages_tree';
        if (isset($params['ul_class'])) {
            $ul_class_name = $ul_class = $params['ul_class'];
        }
        $content_link_class = 'mw-tree-content-link';
        if (isset($params['content_link_class'])) {
            $content_link_class = $params['content_link_class'];
        }

        $li_class = 'pages_tree_item';
        if (isset($params['li_class'])) {
            $li_class = $params['li_class'];
        }
        if (isset($params['ul_tag'])) {
            $list_tag = $params['ul_tag'];
        }
        if (isset($params['li_tag'])) {
            $list_item_tag = $params['li_tag'];
        }
        if (isset($params['include_categories'])) {
            $include_categories = $params['include_categories'];
        }
        $include_all_content = false;
        if (isset($params['include_all_content'])) {
            $include_all_content = $params['include_all_content'];
        }
        ob_start();

        $table = 'content';
        $par_q = '';
        if ($parent == false) {
            $parent = (0);
        } else {
            $parent = intval($parent);
            $par_q = " parent=$parent    and  ";
        }

        if ($include_first == true) {
            $content_type_q = " and content_type='page'  ";
            if ($include_all_content) {
                $content_type_q = ' ';
            }

            $sql = "SELECT * from $table where  id={$parent}    and   is_deleted=0 " . $content_type_q . $is_shop . '  order by position desc  limit 0,1';
        } else {
            $content_type_q = "  content_type='page'  ";
            if ($include_all_content) {
                $content_type_q = ' ';
            }

            $sql = "SELECT * from $table where  " . $par_q . $content_type_q . "   and   is_deleted=0 $is_shop  order by position desc limit 0,100";
        }

        $cid = __FUNCTION__ . crc32($sql);
        $cidg = 'content/' . $parent;
        if (!is_array($params)) {
            $params = array();
        }
        if (isset($append_to_link) == false) {
            $append_to_link = '';
        }
        if (isset($id_prefix) == false) {
            $id_prefix = '';
        }

        if (isset($link) == false) {
            $link = '<span data-page-id="{id}" class="pages_tree_link {nest_level} {active_class} {active_parent_class}" href="{link}' . $append_to_link . '">{title}</span>';
        }

        if (isset($list_tag) == false) {
            $list_tag = 'ul';
        }

        if (isset($active_code_tag) == false) {
            $active_code_tag = '';
        }

        if (isset($list_item_tag) == false) {
            $list_item_tag = 'li';
        }

        if (isset($params['remove_ids'])) {
            $remove_ids = $params['remove_ids'];
        }

        if (isset($remove_ids) and is_string($remove_ids)) {
            $remove_ids = explode(',', $remove_ids);
        }

        if (isset($active_ids)) {
            $active_ids = $active_ids;
        }

        if (isset($active_ids) and is_string($active_ids)) {
            $active_ids = explode(',', $active_ids);
            if (is_array($active_ids) == true) {
                foreach ($active_ids as $idk => $idv) {
                    $active_ids[$idk] = intval($idv);
                }
            }
        }

        $the_active_class = 'active';
        if (isset($params['active_class'])) {
            $the_active_class = $params['active_class'];
        }

        if (!$include_all_content) {
            $params['content_type'] = 'page';
        }

        $include_first_set = false;

        if ($include_first == true) {
            $include_first_set = 1;
            $include_first = false;
            $include_first_set = $parent;
            if (isset($params['include_first'])) {
                unset($params['include_first']);
            }
        } else {
            $params['parent'] = $parent;
        }
        if (isset($params['is_shop']) and $params['is_shop'] == 1) {
            if (isset($params['parent']) and $params['parent'] == 0) {
                unset($params['parent']);
            }

            if (isset($params['parent']) and $params['parent'] == 'any') {
                unset($params['parent']);
            }
        } else {
            if (isset($params['parent']) and $params['parent'] == 'any') {
                $params['parent'] = 0;
            }
        }

        $params['limit'] = 500;
        $params['orderby'] = 'position desc';
        $params['curent_page'] = 1;
        $params['is_deleted'] = 0;
        $params['cache_group'] = false;
        $params['no_cache'] = true;

        $skip_pages_with_no_categories = false;
        $skip_pages_from_tree = false;

        if (isset($params['skip_sub_pages']) and $params['skip_sub_pages'] != '') {
            $skip_pages_from_tree = $params['skip_sub_pages'];
        }
        if (isset($params['skip-static-pages']) and $params['skip-static-pages'] != false) {
            $skip_pages_with_no_categories = 1;
        }

        $params2 = $params;

        if (isset($params2['id'])) {
            unset($params2['id']);
        }
        if (isset($params2['link'])) {
            unset($params2['link']);
        }

        if ($include_first_set != false) {
            $q = $this->get('id=' . $include_first_set);
        } else {
            $q = $this->get($params2);
        }
        $result = $q;
        if (is_array($result) and !empty($result)) {
            ++$nest_level;
            if (trim($list_tag) != '') {
                if ($ul_class_name == false) {
                    echo "<{$list_tag} class='pages_tree depth-{$nest_level}'>";
                } else {
                    echo "<{$list_tag} class='{$ul_class_name} depth-{$nest_level}'>";
                }
            }
            $res_count = 0;
            foreach ($result as $item) {
                if (is_array($item) != false and isset($item['title']) and $item['title'] != null) {
                    $skip_me_cause_iam_removed = false;
                    if (is_array($remove_ids) == true) {
                        foreach ($remove_ids as $idk => $idv) {
                            $remove_ids[$idk] = intval($idv);
                        }

                        if (in_array($item['id'], $remove_ids)) {
                            $skip_me_cause_iam_removed = true;
                        }
                    }

                    if ($skip_pages_with_no_categories == true) {
                        if (isset($item ['subtype']) and $item ['subtype'] != 'dynamic') {
                            $skip_me_cause_iam_removed = true;
                        }
                    }
                    if ($skip_me_cause_iam_removed == false) {
                        $output = $output . $item['title'];
                        $content_type_li_class = false;
                        switch ($item ['subtype']) {
                            case 'dynamic' :
                                $content_type_li_class = 'have_category';
                                break;
                            case 'module' :
                                $content_type_li_class = 'is_module';
                                break;
                            default :
                                $content_type_li_class = 'is_page';
                                break;
                        }

                        if (isset($item ['layout_file']) and stristr($item ['layout_file'], 'blog')) {
                            $content_type_li_class .= ' is_blog';
                        }

                        if ($item['is_home'] == 1) {
                            $content_type_li_class .= ' is_home';
                        }
                        $st_str = '';
                        $st_str2 = '';
                        $st_str3 = '';
                        if (isset($item['subtype']) and trim($item['subtype']) != '') {
                            $st_str = " data-subtype='{$item['subtype']}' ";
                        }

                        if (isset($item['subtype_value']) and trim($item['subtype_value']) != '') {
                            $st_str2 = " data-subtype-value='{$item['subtype_value']}' ";
                        }

                        if (isset($item['is_shop']) and trim($item['is_shop']) == 1) {
                            $st_str3 = ' data-is-shop=true ';
                            $content_type_li_class .= ' is_shop';
                        }
                        $iid = $item['id'];

                        $to_pr_2 = "<{$list_item_tag} class='{$li_class} $content_type_li_class {active_class} {active_parent_class} depth-{$nest_level} item_{$iid} {exteded_classes} menu-item-id-{$item['id']}' data-page-id='{$item['id']}' value='{$item['id']}'  data-item-id='{$item['id']}'  {active_code_tag} data-parent-page-id='{$item['parent']}' {$st_str} {$st_str2} {$st_str3}  title='" . addslashes($item['title']) . "' >";

                        if ($link != false) {
                            $active_parent_class = '';
                            if (intval($item['parent']) != 0 and intval($item['parent']) == intval(main_page_id())) {
                                $active_parent_class = 'active-parent';
                            } elseif (intval($item['id']) == intval(main_page_id())) {
                                $active_parent_class = 'active-parent';
                            } else {
                                $active_parent_class = '';
                            }

                            if ($item['id'] == content_id()) {
                                $active_class = 'active';
                            } elseif (isset($active_ids) and !is_array($active_ids) and $item['id'] == $active_ids) {
                                $active_class = 'active';
                            }
                            if (isset($active_ids) and is_array($active_ids) and in_array($item['id'], $active_ids)) {
                                $active_class = 'active';
                            } elseif ($item['id'] == page_id()) {
                                $active_class = 'active';
                            } elseif ($item['id'] == post_id()) {
                                $active_class = 'active';
                            } elseif (category_id() != false and intval($item['subtype_value']) != 0 and $item['subtype_value'] == category_id()) {
                                $active_class = 'active';
                            } else {
                                $active_class = '';
                            }

                            $ext_classes = '';
                            if ($res_count == 0) {
                                $ext_classes .= ' first-child ';
                                $ext_classes .= ' child-' . $res_count . '';
                            } elseif (!isset($result[$res_count + 1])) {
                                $ext_classes .= ' last-child';
                                $ext_classes .= ' child-' . $res_count . '';
                            } else {
                                $ext_classes .= ' child-' . $res_count . '';
                            }

                            if (isset($item['parent']) and intval($item['parent']) > 0) {
                                $ext_classes .= ' have-parent';
                            }

                            if (isset($item['subtype_value']) and intval($item['subtype_value']) != 0) {
                                $ext_classes .= ' have-category';
                            }

                            if (isset($item['is_active']) and $item['is_active'] == 'n') {
                                $ext_classes = $ext_classes . ' content-unpublished ';
                            }

                            $ext_classes = trim($ext_classes);
                            $the_active_class = $active_class;


                            if (is_callable($link)) {
                                $to_print = call_user_func_array($link, array($item));
                            } else {
                                $to_print = $link;
                            }


                            $to_print = str_replace('{id}', $item['id'], $to_print);
                            $to_print = str_replace('{active_class}', $active_class, $to_print);
                            $to_print = str_replace('{active_parent_class}', $active_parent_class, $to_print);
                            $to_print = str_replace('{exteded_classes}', $ext_classes, $to_print);

                            $to_pr_2 = str_replace('{exteded_classes}', $ext_classes, $to_pr_2);
                            $to_pr_2 = str_replace('{active_class}', $active_class, $to_pr_2);
                            $to_pr_2 = str_replace('{active_parent_class}', $active_parent_class, $to_pr_2);

                            $to_print = str_replace('{title}', $item['title'], $to_print);
                            $to_print = str_replace('{nest_level}', 'depth-' . $nest_level, $to_print);
                            $to_print = str_replace('{content_link_class}', $content_link_class, $to_print);
                            $to_print = str_replace('{empty}', '', $to_print);

                            if (strstr($to_print, '{link}')) {
                                $to_print = str_replace('{link}', page_link($item['id']), $to_print);
                            }

                            $empty1 = intval($nest_level);
                            $empty = '';
                            for ($i1 = 0; $i1 < $empty1; ++$i1) {
                                $empty = $empty . '&nbsp;&nbsp;';
                            }
                            $to_print = str_replace('{empty}', $empty, $to_print);

                            if (strstr($to_print, '{tn}')) {
                                $content_img = get_picture($item['id']);
                                if ($content_img) {
                                    $to_print = str_replace('{tn}', $content_img, $to_print);
                                } else {
                                    $to_print = str_replace('{tn}', '', $to_print);
                                }
                            }
                            foreach ($item as $item_k => $item_v) {
                                if (!is_string($item_k) || !is_string($item_v)) {
                                    continue;
                                }
                                $to_print = str_replace('{' . $item_k . '}', $item_v, $to_print);
                            }
                            ++$res_count;
                            if (isset($active_ids) and is_array($active_ids) == true) {
                                $is_there_active_ids = false;
                                foreach ($active_ids as $active_id) {
                                    if (intval($item['id']) == intval($active_id)) {
                                        $is_there_active_ids = true;
                                        $to_print = str_ireplace('{active_code}', $active_code, $to_print);
                                        $to_print = str_ireplace('{active_class}', $the_active_class, $to_print);
                                        $to_pr_2 = str_ireplace('{active_class}', $the_active_class, $to_pr_2);
                                        $to_pr_2 = str_ireplace('{active_code_tag}', $active_code_tag, $to_pr_2);
                                        $to_pr_2 = str_replace('{empty}', '', $to_pr_2);

                                    }
                                }
                            } elseif (isset($active_ids) and !is_array($active_ids)) {
                                if (intval($item['id']) == intval($active_ids)) {
                                    $is_there_active_ids = true;
                                    $to_print = str_ireplace('{active_code}', $active_code, $to_print);
                                    $to_print = str_ireplace('{active_class}', $the_active_class, $to_print);
                                    $to_pr_2 = str_ireplace('{active_class}', $the_active_class, $to_pr_2);
                                    $to_pr_2 = str_ireplace('{active_code_tag}', $active_code_tag, $to_pr_2);
                                    $to_pr_2 = str_replace('{empty}', '', $to_pr_2);

                                }
                            }

                            $to_print = str_ireplace('{active_code}', '', $to_print);
                            $to_print = str_ireplace('{active_class}', '', $to_print);
                            $to_pr_2 = str_ireplace('{active_class}', '', $to_pr_2);
                            $to_pr_2 = str_ireplace('{active_code_tag}', '', $to_pr_2);
                            $to_pr_2 = str_ireplace('{content_link_class}', '', $to_pr_2);
                            $to_pr_2 = str_replace('{empty}', '', $to_pr_2);

                            $to_print = str_replace('{exteded_classes}', '', $to_print);
                            $to_print = str_replace('{content_link_class}', '', $to_print);

                            $to_print = str_replace('{empty}', '', $to_print);


                            if ($item['id'] == $item['parent']) {
                                $remove_ids[] = intval($item['id']);
                            }

                            if (is_array($remove_ids) == true) {
                                if (in_array($item['id'], $remove_ids)) {
                                    if ($removed_ids_code == false) {
                                        $to_print = false;
                                    } else {
                                        $remove_ids[] = intval($item['id']);
                                        $to_print = str_ireplace('{removed_ids_code}', $removed_ids_code, $to_print);
                                    }
                                } else {
                                    $to_print = str_ireplace('{removed_ids_code}', '', $to_print);
                                }
                            }
                            $to_pr_2 = str_replace('{active_class}', '', $to_pr_2);
                            $to_pr_2 = str_replace('{exteded_classes}', '', $to_pr_2);

                            echo $to_pr_2;
                            $to_pr_2 = false;
                            echo $to_print;
                        } else {
                            $to_pr_2 = str_ireplace('{active_class}', '', $to_pr_2);
                            $to_pr_2 = str_replace('{exteded_classes}', '', $to_pr_2);
                            $to_pr_2 = str_replace('{active_parent_class}', '', $to_pr_2);

                            echo $to_pr_2;
                            $to_pr_2 = false;
                            echo $item['title'];
                        }

                        if (is_array($params)) {
                            $params['parent'] = $item['id'];
                            if ($max_level != false) {
                                $params['max_level'] = $max_level;
                            }
                            if (isset($params['is_shop'])) {
                                unset($params['is_shop']);
                            }

                            //   $nest_level++;
                            $params['nest_level'] = $nest_level;
                            $params['ul_class_name'] = false;
                            $params['ul_class'] = false;
                            if (isset($include_categories)) {
                                $params['include_categories'] = $include_categories;
                            }

                            if (isset($params['ul_class_deep'])) {
                                $params['ul_class'] = $params['ul_class_deep'];
                            }

                            if (isset($maxdepth)) {
                                $params['maxdepth'] = $maxdepth;
                            }

                            if (isset($params['li_class_deep'])) {
                                $params['li_class'] = $params['li_class_deep'];
                            }

                            if (isset($params['return_data'])) {
                                unset($params['return_data']);
                            }

                            $params['remove_ids'] = $remove_ids;
                            if ($skip_pages_from_tree == false) {
                                if ($item['id'] != $item['parent']) {
                                    $children = $this->pages_tree($params);
                                }
                            }
                        } else {
                            if ($skip_pages_from_tree == false) {
                                if ($item['id'] != $item['parent']) {
                                    $children = $this->pages_tree(intval($item['id']), $link, $active_ids, $active_code, $remove_ids, $removed_ids_code, $ul_class_name = false);
                                }
                            }
                        }

                        if (isset($include_categories) and $include_categories == true) {
                            $content_cats = array();
                            if (isset($item['subtype_value']) and intval($item['subtype_value']) == true) {
                            }

                            $cat_params = array();
                            if (isset($item['subtype_value']) and intval($item['subtype_value']) != 0) {
                                //$cat_params['subtype_value'] = $item['subtype_value'];
                            }
                            //$cat_params['try_rel_id'] = $item['id'];

                            if (isset($categories_link)) {
                                $cat_params['link'] = $categories_link;
                            } else {
                                $cat_params['link'] = $link;
                            }

                            if (isset($categories_active_ids)) {
                                $cat_params['active_ids'] = $categories_active_ids;

                            }

                            if (isset($categories_removed_ids)) {
                                $cat_params['remove_ids'] = $categories_removed_ids;
                            }

                            if (isset($active_code)) {
                                $cat_params['active_code'] = $active_code;
                            }

                            if (isset($params['categories_extra_attributes'])) {
                                $cat_params['extra_attributes'] = $params['categories_extra_attributes'];
                            }


                            //$cat_params['for'] = 'content';
                            $cat_params['list_tag'] = $list_tag;
                            $cat_params['list_item_tag'] = $list_item_tag;
                            $cat_params['rel_type'] = 'content';
                            $cat_params['rel_id'] = $item['id'];

                            $cat_params['include_first'] = 1;
                            $cat_params['nest_level'] = $nest_level + 1;
                            if ($max_level != false) {
                                $cat_params['max_level'] = $max_level;
                            }

                            if ($nest_level > 1) {
                                if (isset($params['ul_class_deep'])) {
                                    $cat_params['ul_class'] = $params['ul_class_deep'];
                                }
                                if (isset($params['li_class_deep'])) {
                                    $cat_params['li_class'] = $params['li_class_deep'];
                                }
                            } else {
                                if (isset($params['ul_class'])) {
                                    $cat_params['ul_class'] = $params['ul_class'];
                                }
                                if (isset($params['li_class'])) {
                                    $cat_params['li_class'] = $params['li_class'];
                                }
                            }


                            if (isset($params['categories_ul_class'])) {
                                $cat_params['ul_class'] = $params['categories_ul_class'];
                            }

                            if (isset($params['categories_link_class'])) {
                                $cat_params['link_class'] = $params['categories_link_class'];
                            }


                            if (isset($params['categories_li_class'])) {
                                $cat_params['li_class'] = $params['categories_li_class'];
                            }
                            if (isset($params['categories_ul_class_deep'])) {
                                $cat_params['ul_class_deep'] = $params['categories_ul_class_deep'];
                            }


                            if (isset($params['categories_li_class_deep'])) {
                                $cat_params['li_class_deep'] = $params['categories_li_class_deep'];
                            }

                            if (isset($params['active_class'])) {
                                $cat_params['active_class'] = $params['active_class'];
                            }

                            $this->app->category_manager->tree($cat_params);
                        }
                    }
                    echo "</{$list_item_tag}>";
                }
            }
            if (trim($list_tag) != '') {
                echo "</{$list_tag}>";
            }
        }
        $content = ob_get_contents();
        if ($nest_level_orig == 0) {
            //     $this->app->cache_manager->save($content, $function_cache_id, $cache_group);
        }

        if (isset($list_item_tag) and $list_item_tag and $list_item_tag == 'option') {
            $content = str_replace('</option></option>', '</option>', $content);
        }

        ob_end_clean();
        if (isset($params['return_data'])) {
            return $content;
        } else {
            echo $content;
        }

        return false;
    }

    /**
     * Defines all constants that are needed to parse the page layout.
     *
     * It accepts array or $content that must have  $content['id'] set
     *
     * @param array|bool $content
     *
     * @option     integer  "id"   [description]
     * @option     string "content_type" [description]
     * @example
     * <code>
     *  Define constants for some page
     *  $ref_page = $this->get_by_id(1);
     *  $this->define_constants($ref_page);
     *  print PAGE_ID;
     *  print POST_ID;
     *  print CATEGORY_ID;
     *  print MAIN_PAGE_ID;
     *  print DEFAULT_TEMPLATE_DIR;
     *  print DEFAULT_TEMPLATE_URL;
     * </code>
     *
     * @const      PAGE_ID Defines the current page id
     * @const      POST_ID Defines the current post id
     * @const      CATEGORY_ID Defines the current category id if any
     * @const      ACTIVE_PAGE_ID Same as PAGE_ID
     * @const      CONTENT_ID current post or page id
     * @const      MAIN_PAGE_ID the parent page id
     * @const      DEFAULT_TEMPLATE_DIR the directory of the site's default template
     * @const      DEFAULT_TEMPLATE_URL the url of the site's default template
     *
     */
    public function define_constants($content = false)
    {
        return $this->app->template_manager->defineConstants($content);
    }


    /**
     *  Get the first parent that has layout.
     */
    public function get_inherited_parent($content_id)
    {

        return app()->content_repository->getInheritedParent($content_id);
    }


    public function get_parents($id = 0)
    {
        return app()->content_repository->getParents($id);

    }

    /**
     * Get the current language of the site.
     *
     * @example
     * <code>
     *  $current_lang = current_lang();
     *  print $current_lang;
     * </code>
     *
     * @constant  MW_LANG defines the MW_LANG constant
     */
//    public function lang_current()
//    {
//        if (defined('MW_LANG') and MW_LANG != false) {
//            return MW_LANG;
//        }
//
//        $lang = false;
//
//        if (!isset($lang) or $lang == false) {
//            if (isset($_COOKIE['lang'])) {
//                $lang = $_COOKIE['lang'];
//            }
//        }
//        if (!isset($lang) or $lang == false) {
//            $def_language = $this->app->option_manager->get('language', 'website');
//            if ($def_language != false) {
//                $lang = $def_language;
//            }
//        }
//        if (!isset($lang) or $lang == false) {
//            $lang = 'en';
//        }
//        $lang = sanitize_path($lang);
//        if (!defined('MW_LANG') and isset($lang)) {
//            define('MW_LANG', $lang);
//        }
//
//        return $lang;
//    }

    /**
     * Set the current language.
     *
     * @example
     * <code>
     *   //sets language to Spanish
     *  set_language('es');
     * </code>
     */
//    public function lang_set($lang = 'en')
//    {
//        $lang = sanitize_path($lang);
//        setcookie('lang', $lang);
//
//        return $lang;
//    }

    public function breadcrumb($params = false)
    {
        return $this->breadcrumbLinks->get($params);
    }

    /**
     * Gets a link for given content id.
     *
     * If you don't pass id parameter it will try to use the current page id
     *
     * @param int $id The $id The id of the content
     *
     * @return string The url of the content
     *
     * @see     post_link()
     * @see     page_link()
     * @see     content_link()
     *
     * @example
     * <code>
     * print $this->link($id=1);
     * </code>
     */
    public function link($id = 0)
    {
        if (is_array($id)) {
            extract($id);
        }

        if ($id == false or $id == 0) {
            $id = content_id();
        }

        if ($id == 0) {
            return $this->app->url_manager->site();
        }

        $link = $this->get_by_id($id);
        if (!$link) {
            return;
        }

        $site_url = $this->app->url_manager->site();

        if (isset($link['is_home']) and intval($link['is_home']) == 1) {
            return $site_url;
        }

        if (!isset($link['url']) or strval($link['url']) == '') {
            $link = $this->get_by_url($id);
        }
        if (!$link) {
            return;
        }

        $permalinkGenerated = $this->app->permalink_manager->link($link['id'], 'content');

        if ($permalinkGenerated) {
            $link['url'] = $permalinkGenerated;

            if (!stristr($link['url'], $site_url)) {
                $link = site_url($link['url']);
            } else {
                $link = ($link['url']);
            }
            return $link;
        }

    }

    public function create_link($contentType = 'page')
    {

        if ($contentType == 'product') {
            return admin_url('products/create');
        }

        if ($contentType == 'post') {
            return admin_url('posts/create');
        }

        if ($contentType == 'page') {
            return admin_url('pages/create');
        }

        return admin_url('content/create');
    }

    public function edit_link($id = 0)
    {
        $content = $this->get_by_id($id);

        if (isset($content['content_type']) && $content['content_type'] == 'product') {
            // return route('admin.product.edit', $id);
            return admin_url('products/' . $id . '/edit');
        }

        if (isset($content['content_type']) && $content['content_type'] == 'post') {
            // return route('admin.post.edit', $id);
            return admin_url('posts/' . $id . '/edit');
        }

        if (isset($content['content_type']) && $content['content_type'] == 'page') {
            return admin_url('pages/' . $id . '/edit');
            //  return route('admin.page.edit', $id);
        }

        // return route('admin.content.edit', $id);
        return admin_url('content/' . $id . '/edit');

    }

    public function save_edit($post_data)
    {
        return $this->helpers->save_from_live_edit($post_data);
    }

    /**
     * Returns the homepage as array.
     *
     * @category Content
     */
    public function homepage()
    {
        $get = array();
        $get['is_home'] = 1;
        $get['single'] = 1;

        $data = app()->content_repository->getByParams($get);

        return $data;
    }


    public function save_content($data, $delete_the_cache = true)
    {
        if (is_string($data)) {
            $data = parse_params($data);
        }

        $this->app->event_manager->trigger('content.manager.before.save', $data);
        $data_to_save = $data;
        $save = $this->crud->save($data);
        $id = $save;
        if (isset($data_to_save['add_content_to_menu']) and is_array($data_to_save['add_content_to_menu'])) {
            foreach ($data_to_save['add_content_to_menu'] as $menu_id) {
                $ids_to_save = $save;
                if (!app()->menu_manager->is_in_menu($menu_id, $ids_to_save)) {
                    $this->helpers->add_content_to_menu($ids_to_save, $menu_id);
                }
            }
        }
        $after_save = $data_to_save;
        $after_save['id'] = $id;

        $this->app->event_manager->trigger('content.manager.after.save', $after_save);
        event_trigger('mw_save_content', $save);

        return $save;
    }

    public function custom_fields($content_id, $full = true, $field_type = false)
    {
        $filter = [];
        $filter['rel_type'] = morph_name(\Modules\Content\Models\Content::class);
        $filter['rel_id'] = $content_id;
        if ($full) {
            $filter['return_full'] = $full;
        }
        if ($field_type) {
            $filter['type'] = $field_type;
        }

        return $this->app->fields_manager->get($filter);
    }

    public function save_content_field($data, $delete_the_cache = true)
    {
        return $this->helpers->save_content_field($data, $delete_the_cache);
    }

    public function edit_field($data)
    {
        return $this->crud->get_edit_field($data);
    }

    public function save($data, $delete_the_cache = true)
    {
        return $this->save_content($data, $delete_the_cache);
    }

    public function prev_content($content_id = false)
    {
        return $this->next_content($content_id, $mode = 'prev');
    }

    public function next_content($content_id = false, $mode = 'next', $content_type = false)
    {
        if ($content_id == false) {
            if (defined('POST_ID') and POST_ID != 0) {
                $content_id = POST_ID;
            } elseif (defined('PAGE_ID') and PAGE_ID != 0) {
                $content_id = PAGE_ID;
            } elseif (defined('MAIN_PAGE_ID') and MAIN_PAGE_ID != 0) {
                $content_id = MAIN_PAGE_ID;
            }
        }
        $category_id = false;
        if (defined('CATEGORY_ID') and CATEGORY_ID != 0) {
            $category_id = CATEGORY_ID;
        }

        if ($content_id == false) {
            return false;
        } else {
            $content_id = intval($content_id);
        }
        $contentData = $this->get_by_id($content_id);
        if ($contentData == false) {
            return false;
        }

        if ($contentData['position'] == null) {
            $contentData['position'] = 0;
        }

        $query = \Modules\Content\Models\Content::query()->select('content.*');
        $categories = array();
        $params = array();

        $parent_id = false;
        if (isset($contentData['parent']) and $contentData['parent'] > 0) {
            $parent_id = $contentData['parent'];
        }

        if ($content_type) {
            if (defined('PAGE_ID') and PAGE_ID != 0) {
                $parent_id = PAGE_ID;
            }
        } elseif (isset($contentData['content_type'])) {
            $content_type = $contentData['content_type'];
        }

        if (isset($contentData['content_type']) and $contentData['content_type'] != 'page') {

            if (trim($mode) == 'prev') {
                $query->orderBy('position', 'desc');
                $query->where('position', '<', $contentData['position']);
            } else {
                $query->orderBy('position', 'asc');
                $query->where('position', '>', $contentData['position']);
            }

            $cats = $this->app->category_manager->get_for_content($content_id);
            if (!empty($cats)) {
                foreach ($cats as $cat) {
                    $categories[] = $cat['id'];
                }
                $query->whereCategoryIds($categories);
            }

        } else {

            if (isset($contentData['position']) and $contentData['position'] > 0) {
                if (trim($mode) == 'prev') {
                    $query->where('position', '>', $contentData['position']);
                } else {
                    $query->where('position', '<', $contentData['position']);
                }
            }

            if (trim($mode) == 'prev') {
                $query->orderBy('created_at', 'desc');
            } else {
                $query->orderBy('created_at', 'asc');
            }
        }

        $params['exclude_ids'] = array($content_id);

        if ($parent_id) {
            $query->whereParent($parent_id);
        }

        $query->whereContentType($content_type);
        $query->whereIsActive(1);
        $query->whereIsDeleted(0);

        $response = [];
        $get = $query->first();
        if ($get != null) {
            $response = $get->toArray();
        }

        if (is_array($response)) {
            return $response;
        } else {
            return false;
        }
    }

    public function reorder($params)
    {
        $id = $this->app->user_manager->is_admin();
        if ($id == false) {
            return array('error' => 'You must be admin to reorder content!');
        }

        return $this->crud->reorder($params);
    }

    /**
     * Set content to be unpublished.
     *
     * Set is_active flag 'n'
     *
     * @param string|array|bool $params
     *
     * @return string The url of the content
     *
     * @uses       $this->save_content()
     *
     * @see        content_set_unpublished()
     *
     * @example
     * <code>
     * //set published the content with id 5
     * content_set_unpublished(5);
     *
     * //alternative way
     * content_set_unpublished(array('id' => 5));
     * </code>
     */
    public function set_unpublished($params)
    {
        if (intval($params) > 0 and !isset($params['id'])) {
            if (!is_array($params)) {
                $id = $params;
                $params = array();
                $params['id'] = $id;
            }
        }


        if (!isset($params['id'])) {
            return array('error' => 'You must provide id parameter!');
        } else {
            if (intval($params['id'] != 0)) {
                $save = array();
                $save['id'] = intval($params['id']);
                $save['is_active'] = 0;

                $save_data = $this->save_content($save);

                return $save_data;
            }
        }
    }

    /**
     * Set content to be published.
     *
     * Set is_active flag 'y'
     *
     * @param string|array|bool $params
     *
     * @return string The url of the content
     *
     * @uses       $this->save_content()
     *
     * @example
     * <code>
     * //set published the content with id 5
     * api/content/set_published(5);
     *
     * //alternative way
     * api/content/set_published(array('id' => 5));
     * </code>
     */
    public function set_published($params)
    {
        if (intval($params) > 0 and !isset($params['id'])) {
            if (!is_array($params)) {
                $id = $params;
                $params = array();
                $params['id'] = $id;
            }
        }
//admin check is moved to middelware
//        $adm = $this->app->user_manager->is_admin();
//        if ($adm == false) {
//            return array('error' => 'You must be admin to publish content!');
//        }

        if (!isset($params['id'])) {
            return array('error' => 'You must provide id parameter!');
        } else {
            if (intval($params['id'] != 0)) {
                $save = array();
                $save['id'] = intval($params['id']);
                $save['is_active'] = 1;

                $save_data = $this->save_content($save);

                return $save_data;
            }
        }
    }

    public function save_content_data_field($data, $delete_the_cache = true)
    {
        return $this->app->data_fields_manager->save($data);
    }

    public function get_pages($params = false)
    {
        $params2 = array();

        if (is_string($params)) {
            $params = parse_str($params, $params2);
            $params = $params2;
        }

        if (!is_array($params)) {
            $params = array();
        }
        if (!isset($params['content_type'])) {
            $params['content_type'] = 'page';
        }


        return $this->get($params);
    }

    public function get_posts($params = false)
    {
        $params2 = array();
        if (is_string($params)) {
            $params = parse_str($params, $params2);
            $params = $params2;
        }
        if (!is_array($params)) {
            $params = array();
        }
        if (!isset($params['content_type'])) {
            $params['content_type'] = 'post';
        }
        if (!isset($params['subtype'])) {
            $params['subtype'] = 'post';
        }

        return $this->get($params);
    }

    public function get_products($params = false)
    {
        if (is_string($params)) {
            $params = parse_params($params);
        }
        if (!is_array($params)) {
            $params = array();
        }
        if (!isset($params['content_type'])) {
            $params['content_type'] = 'product';
        }

        return $this->get($params);
    }

    public function title($id)
    {


        if ($id == false or $id == 0) {
            if (defined('CONTENT_ID') == true) {
                $id = CONTENT_ID;
            } elseif (defined('PAGE_ID') == true) {
                $id = PAGE_ID;
            }
        }
        $content = $this->get_by_id($id);
        if (isset($content['title'])) {
            return $content['title'];
        }
    }


    public function description($id)
    {
        $descr = false;

        if ($id == false or $id == 0) {
            if (defined('CONTENT_ID') == true) {
                $id = CONTENT_ID;
            } elseif (defined('PAGE_ID') == true) {
                $id = PAGE_ID;
            }
        }
        $meta = $this->get_by_id($id);
        if (!$meta) {
            return;
        }

        if (isset($meta['description']) and $meta['description'] != '') {
            $descr = $meta['description'];
        } else if (isset($meta['content_meta_description']) and $meta['content_meta_description'] != '') {
            $descr = $meta['content_meta_description'];
        } else if (isset($meta['content_body']) and $meta['content_body'] != '') {
            $descr = strip_tags($meta['content_body']);
        } else if (isset($meta['content']) and $meta['content'] != '') {
            $descr = strip_tags($meta['content']);
        }

        if ($descr) {
            $descr = trim($descr);
        }
        if ($descr) {
            if ($descr == 'mw_saved_inner_edit_from_parent_edit_field') {
                $descr_from_edit_field = $this->app->content_manager->edit_field("rel_type=content&rel_id=" . $id);
                if ($descr_from_edit_field) {
                    $descr_from_edit_field = trim(strip_tags($descr_from_edit_field));
                }
                if ($descr_from_edit_field) {
                    $descr = trim($descr_from_edit_field);
                }

            }
        }
        if ($descr) {
            return $descr;
        }
    }


    public function get_related_content_ids_for_content_id($content_id = false)
    {
        return $this->app->content_repository->getRelatedContentIds($content_id);

    }


}
